
# Move Task Start Dates
# Copyright 2006 by Alexander V. Christensen

"""
Shift the start dates of the selected tasks
  by a given number of days, weeks, months, or years.
"""

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

def hint(s):
    try:
        Data.Hint("%s: %s" % (scriptname, s))
    except AttributeError:
        self.SetStatusText(s)

defUnitTypes = [('Days', 'Day', 1), ('Weeks', 'Day', 7),
    ('Months', 'Month', 1), ('Years', 'Month', 12)]

class DateShiftDialog(wx.Dialog):
    def __init__(self, parent, unitTypes=defUnitTypes):
        wx.Dialog.__init__(self, parent, -1, 'Adjust Dates')
        self.unitChoices = []
        self.unitTypes = {}
        for unit, type, size in unitTypes:
            self.unitChoices.append(unit)
            self.unitTypes[unit] = type, size

        border_sizer = wx.BoxSizer(wx.VERTICAL)
        sizer = wx.BoxSizer(wx.VERTICAL)

        controls = wx.BoxSizer(wx.HORIZONTAL)

        self.sizeCtrl = wx.TextCtrl(self, -1, size=(60, 25))
        size = Data.Option.get('DateShiftSize', '1')
        self.sizeCtrl.SetValue(size)
        self.sizeCtrl.Bind(wx.EVT_SET_FOCUS, self.OnFocus)
        controls.Add(self.sizeCtrl, 0, wx.LEFT | wx.RIGHT | wx.ALIGN_CENTRE, 5)

        self.unitCtrl = wx.Choice(self, -1, choices=self.unitChoices)
        unit = Data.Option.get('DateShiftUnit')
        if not unit or not self.unitCtrl.SetStringSelection(unit):
            self.unitCtrl.SetSelection(0)
        controls.Add(self.unitCtrl, 0, wx.LEFT | wx.RIGHT | wx.ALIGN_CENTRE, 5)

        self.aimCtrl = wx.Choice(self, -1, choices=['Forward', 'Backward'])
        aim = Data.Option.get('DateShiftAim')
        if not aim or not self.aimCtrl.SetStringSelection(aim):
            self.aimCtrl.SetSelection(0)
        controls.Add(self.aimCtrl, 0, wx.LEFT | wx.RIGHT | wx.ALIGN_CENTRE, 5)
        sizer.Add(controls, 0, wx.ALIGN_CENTRE)
        sizer.Add((25, 25))

        buttons = wx.BoxSizer(wx.HORIZONTAL)

        cancelbutton = wx.Button(self, wx.ID_CANCEL, "Cancel", size=(70, 20))
        buttons.Add(cancelbutton, 0, wx.RIGHT, 5)

        okbutton = wx.Button(self, wx.ID_OK, "OK", size=(70, 20))
        okbutton.SetDefault()
        okbutton.Bind(wx.EVT_BUTTON, self.OnOkay)
        buttons.Add(okbutton, 0, wx.LEFT, 5)

        sizer.Add(buttons, 0, wx.ALIGN_RIGHT)

        border_sizer.Add(sizer, 0, wx.ALL | wx.ALIGN_CENTRE, 10)
        self.SetSizer(border_sizer)
        self.Fit()
        self.CentreOnScreen()

    def OnFocus(self, evt):
        source = evt.GetEventObject()
        if isinstance(source, wx.TextCtrl):
             source.SetSelection(-1, -1)
        evt.Skip()
        self.Refresh()

    def GetDateShift(self):
        size = self.sizeCtrl.GetValue()
        try:
            size = int(size)
        except ValueError:
            size = 0
        unitString = self.unitCtrl.GetStringSelection()
        unitType, unitSize = self.unitTypes[unitString]
        size *= unitSize
        if self.aimCtrl.GetSelection() == 1:  # backwards move
            size *= -1
        return unitType, size

    def OnOkay(self, evt):
        Data.Option['DateShiftSize'] = self.sizeCtrl.GetValue()
        Data.Option['DateShiftUnit'] = self.unitCtrl.GetStringSelection()
        Data.Option['DateShiftAim'] = self.aimCtrl.GetStringSelection()
        Data.SaveOption()
        evt.Skip()

def Do():
    self.SetStatusText("")

    if not isinstance(self.Report, wx.grid.Grid):
        hint("Run this script from a task report.")
        return

    sel = self.Report.GetSelectedRows()
    rowids = self.Report.table.rows  # if nothing selected, use all rows
    if sel:
        rowids = [rowids[s] for s in sel]
    rows = [Data.ReportRow.get(rid, {}) for rid in rowids]
    tasks = [r.get('TableID') for r in rows if r.get('TableName') == 'Task']

    if not tasks:
        rt = self.Report.table.reporttype
        ta, tb = rt.get('TableA'), rt.get('TableB')
        if ta == 'Task' or tb == 'Task':
            if sel:
                hint("Select one or more task rows.")
            else:
                hint("This report has no tasks yet.")
        else:
            hint("Run this script from a task report.")
        return

    for x in tasks:
        task = Data.Task.get(x, {})
        startdate = task.get('StartDate')
        if Data.ValidDate(startdate):
            break
    else:
        hint("None of the selected tasks had a start date.")
        return

    dlg = DateShiftDialog(None)
    if dlg.ShowModal() != wx.ID_OK:
        return
    type, size = dlg.GetDateShift()

    for x in tasks:
        task = Data.Task.get(x, {})
        startdate = task.get('StartDate')
        if not Data.ValidDate(startdate):
            continue
        date = Data.DateConv[startdate]
        try:
            date = Data.AddPeriod(date, type, size)
        except AttributeError:
            date = Data.GetPeriodStart(type, date, size)
        startdate = Data.DateIndex[date]
        change = {'Table': 'Task', 'ID': x, 'StartDate': startdate}
        Data.Update(change)
    Data.SetUndo('Move Task Start Dates')

Do()
